<?php
// atualizar-msi.php — Atualizador do setup.msi/.exe com layout unificado + streaming/log e upload com progresso
// Requer: src/_condb.php com conDb() e permissão de escrita na pasta do script.

ob_start();
session_start();
date_default_timezone_set('America/Sao_Paulo');

require_once __DIR__ . '/src/_condb.php';

// Caminhos candidatos
define('MSI_REL', 'setup.msi');
define('EXE_REL', 'setup.exe');
define('MSI_PATH', __DIR__ . '/' . MSI_REL);
define('EXE_PATH', __DIR__ . '/' . EXE_REL);

// Para o fluxo via LINK, continuamos baixando como MSI por padrão
$fileRel  = MSI_REL;
$filePath = MSI_PATH;

/* -----------------------------
   SCHEMA: cria se não existir
------------------------------*/
function ensure_schema(PDO $pdo){
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS msi_config (
          id             INTEGER PRIMARY KEY CHECK (id=1),
          current_file   TEXT,
          current_size   INTEGER,
          current_sha256 TEXT,
          current_mtime  INTEGER,
          last_file      TEXT,
          last_size      INTEGER,
          last_sha256    TEXT,
          last_mtime     INTEGER,
          download_url   TEXT,
          proxy_scheme   TEXT,
          proxy_host     TEXT,
          proxy_port     INTEGER,
          proxy_user     TEXT,
          proxy_pass     TEXT,
          last_log_id    INTEGER,
          updated_at     TEXT
        );
    ");
    $pdo->exec("INSERT OR IGNORE INTO msi_config(id) VALUES (1);");

    $pdo->exec("
        CREATE TABLE IF NOT EXISTS update_logs (
          id          INTEGER PRIMARY KEY AUTOINCREMENT,
          started_at  TEXT,
          finished_at TEXT,
          ok          INTEGER,
          messages    TEXT
        );
    ");
}

/* -----------------------------
   Utils de arquivo ativo
------------------------------*/
function pick_active_file(): array {
    $cands = [];
    if (is_file(MSI_PATH)) $cands[] = ['rel'=>MSI_REL, 'path'=>MSI_PATH, 'mtime'=>@filemtime(MSI_PATH) ?: 0];
    if (is_file(EXE_PATH)) $cands[] = ['rel'=>EXE_REL, 'path'=>EXE_PATH, 'mtime'=>@filemtime(EXE_PATH) ?: 0];
    if (!$cands) return ['rel'=>MSI_REL, 'path'=>MSI_PATH, 'mtime'=>null, 'present'=>false];
    usort($cands, function($a,$b){ return ($b['mtime'] <=> $a['mtime']) ?: strcmp($a['rel'],$b['rel']); });
    $top = $cands[0];
    $top['present'] = true;
    return $top;
}

/* -----------------------------
   HASH (com cache leve)
------------------------------*/
function sha256_cached(string $path): array {
    $mtime = @filemtime($path);
    if ($mtime === false) return [null, null];

    $cache = $path . '.sha256.json';
    $cached = @file_get_contents($cache);
    if ($cached) {
        $obj = json_decode($cached, true);
        if (is_array($obj) && ($obj['mtime'] ?? null) === $mtime && !empty($obj['sha256'])) {
            return [$obj['sha256'], $mtime];
        }
    }
    $hash = @hash_file('sha256', $path);
    if ($hash) {
        @file_put_contents($cache . '.tmp', json_encode(['mtime'=>$mtime,'sha256'=>$hash], JSON_UNESCAPED_SLASHES), LOCK_EX);
        @rename($cache . '.tmp', $cache);
    }
    return [$hash, $mtime];
}

function get_dl_user_agent(): string {
    $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if ($ua && stripos($ua, 'Mozilla/') !== false) return $ua;
    return 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36';
}

/* -----------------------------
   META do arquivo
------------------------------*/
function get_current_meta(string $filePath, string $fileRel): array {
    if (!is_file($filePath)) {
        return [
            'present' => false,
            'file' => $fileRel,
            'size' => 0,
            'sha256' => null,
            'mtime' => null,
            'last_modified' => null,
        ];
    }
    $size = filesize($filePath);
    [$sha256, $mtime] = sha256_cached($filePath);
    return [
        'present' => true,
        'file' => $fileRel,
        'size' => $size,
        'sha256' => $sha256,
        'mtime' => $mtime,
        'last_modified' => $mtime ? gmdate('D, d M Y H:i:s', $mtime).' GMT' : null,
    ];
}

/* -----------------------------
   Ler/Gravar config
------------------------------*/
function load_cfg(PDO $pdo): array {
    $row = $pdo->query("SELECT * FROM msi_config WHERE id=1")->fetch(PDO::FETCH_ASSOC);
    return $row ?: [];
}
function save_cfg(PDO $pdo, array $set){
    $cols = [
        'download_url','proxy_scheme','proxy_host','proxy_port','proxy_user','proxy_pass',
        'current_file','current_size','current_sha256','current_mtime',
        'last_file','last_size','last_sha256','last_mtime','last_log_id','updated_at'
    ];
    $pairs = []; $vals = [];
    foreach($cols as $c){
        if (array_key_exists($c, $set)) {
            $pairs[] = "$c=?";
            $vals[]  = $set[$c];
        }
    }
    if (!$pairs) return;
    $sql = "UPDATE msi_config SET ".implode(',', $pairs)." WHERE id=1";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($vals);
}

/* -----------------------------
   LOG helpers
------------------------------*/
function log_line(array &$log, string $msg, ?callable $emit=null){
    $line = '['.date('d/m/y H:i:s').'] '.$msg;
    $log[] = $line;
    if ($emit) { echo $line . "\n"; @ob_flush(); @flush(); }
}
function persist_log(PDO $pdo, array $log, bool $ok, ?int $startedAtTs=null): int {
    $started = $startedAtTs ? date('Y-m-d H:i:s', $startedAtTs) : date('Y-m-d H:i:s');
    $finished= date('Y-m-d H:i:s');
    $stmt = $pdo->prepare("INSERT INTO update_logs(started_at, finished_at, ok, messages) VALUES(?,?,?,?)");
    $stmt->execute([$started, $finished, $ok?1:0, implode("\n", $log)]);
    return (int)$pdo->lastInsertId();
}

/* -----------------------------
   Parse de proxy
------------------------------*/
function parse_proxy(?string $s): ?array {
    $s = trim((string)$s);
    if ($s === '') return null;

    if (preg_match('#^(?:(https?|socks5h?|socks5|socks4a?)://)?(?:(.+?):(.*?)@)?([^:/]+):(\d+)$#i', $s, $m)) {
        $scheme = $m[1] ?: 'http';
        $user   = $m[2] ?? '';
        $pass   = $m[3] ?? '';
        $host   = $m[4];
        $port   = (int)$m[5];
        return ['scheme'=>$scheme,'host'=>$host,'port'=>$port,'user'=>$user,'pass'=>$pass];
    }
    $parts = explode(':', $s);
    if (count($parts) === 4) {
        return ['scheme'=>'http','host'=>$parts[0],'port'=>(int)$parts[1],'user'=>$parts[2],'pass'=>$parts[3]];
    }
    if (count($parts) === 2 && ctype_digit($parts[1])) {
        return ['scheme'=>'http','host'=>$parts[0],'port'=>(int)$parts[1],'user'=>'','pass'=>''];
    }
    return null;
}

/* -----------------------------
   Detectar IP do proxy (opcional)
------------------------------*/
function detect_proxy_ip(?array $proxy, array &$log, ?callable $emit=null): ?array {
    if (!$proxy) return null;
    $ch = curl_init('https://ipinfo.io/json');
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPTCONNECTTIMEOUT  => 15,
        CURLOPT_TIMEOUT        => 20,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);
    $proxyUri = $proxy['scheme'].'://'.$proxy['host'].':'.$proxy['port'];
    curl_setopt($ch, CURLOPT_PROXY, $proxyUri);
    if (!empty($proxy['user'])) {
        curl_setopt($ch, CURLOPT_PROXYUSERPWD, $proxy['user'].':'.$proxy['pass']);
    }
    $out = curl_exec($ch);
    if ($out === false) {
        log_line($log, 'Falha ao checar IP pelo proxy: '.curl_error($ch), $emit);
        curl_close($ch);
        return null;
    }
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($code >= 200 && $code < 300) {
        $j = json_decode($out, true);
        if (is_array($j) && !empty($j['ip'])) {
            $ip = $j['ip']; $region = $j['region'] ?? null; $country= $j['country'] ?? null; $city = $j['city'] ?? null;
            log_line($log, "Proxy conectado! IP {$ip} ".($city?"{$city}/":'').($region?$region.' ':'').($country?$country:''), $emit);
            return $j;
        }
    }
    log_line($log, 'Não foi possível determinar IP/região pelo proxy.', $emit);
    return null;
}

/* -----------------------------
   Download (via LINK) c/ progresso
------------------------------*/
function download_file(string $url, string $dstTmp, ?array $proxy, array &$log, ?callable $emit=null, ?string $ua=null): bool {
    $fp = @fopen($dstTmp, 'wb');
    if (!$fp) { log_line($log, "Erro: não foi possível abrir o arquivo temporário: {$dstTmp}", $emit); return false; }

    $ch = curl_init($url);
    $lastPctEmitted = -10;
    $ua = $ua ?: get_dl_user_agent();

    $opts = [
        CURLOPT_FILE            => $fp,
        CURLOPT_FOLLOWLOCATION  => true,
        CURLOPT_MAXREDIRS       => 5,
        CURLOPT_CONNECTTIMEOUT  => 30,
        CURLOPT_TIMEOUT         => 1200,
        CURLOPT_USERAGENT       => $ua,
        CURLOPT_SSL_VERIFYPEER  => true,
        CURLOPT_SSL_VERIFYHOST  => 2,
        CURLOPT_FAILONERROR     => false,
        CURLOPT_NOPROGRESS      => false,
    ];
    if ($proxy) {
        $opts[CURLOPT_PROXY] = $proxy['scheme'].'://'.$proxy['host'].':'.$proxy['port'];
        if (!empty($proxy['user'])) $opts[CURLOPT_PROXYUSERPWD] = $proxy['user'].':'.$proxy['pass'];
    }

    $progressCb = function($resource, $dlTotal, $dlNow) use (&$lastPctEmitted, &$log, $emit) {
        if ($dlTotal > 0) {
            $pct = (int) floor($dlNow * 100 / $dlTotal);
            if ($pct >= $lastPctEmitted + 5) { $lastPctEmitted = $pct; log_line($log, "Download {$pct}% concluído…", $emit); }
        }
        return 0;
    };
    if (defined('CURLOPT_XFERINFOFUNCTION')) $opts[CURLOPT_XFERINFOFUNCTION] = $progressCb;
    else $opts[CURLOPT_PROGRESSFUNCTION] = $progressCb;

    curl_setopt_array($ch, $opts);
    log_line($log, 'Download iniciado…', $emit);

    $ok = curl_exec($ch);
    if ($ok === false) { log_line($log, 'Erro no download: '.curl_error($ch), $emit); curl_close($ch); fclose($fp); @unlink($dstTmp); return false; }
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch); fclose($fp);

    if ($code < 200 || $code >= 300) { log_line($log, "Falha HTTP: código {$code}", $emit); @unlink($dstTmp); return false; }
    log_line($log, 'Download finalizado com sucesso.', $emit);
    return true;
}

/* -----------------------------
   Detecção/validação de MSI/EXE
------------------------------*/
function is_probably_msi(string $path): bool {
    $h = @fopen($path, 'rb'); if (!$h) return false;
    $sig = fread($h, 8); fclose($h);
    if ($sig === false || strlen($sig) < 8) return false;
    $bytes = array_map('ord', str_split($sig, 1));
    $ole = [0xD0,0xCF,0x11,0xE0,0xA1,0xB1,0x1A,0xE1];
    for($i=0;$i<8;$i++){ if ($bytes[$i] !== $ole[$i]) return false; }
    return true;
}
function is_probably_exe(string $path): bool {
    $h = @fopen($path, 'rb'); if (!$h) return false;
    $hdr = fread($h, 64); if ($hdr === false || strlen($hdr) < 64) { fclose($h); return false; }
    if (substr($hdr,0,2) !== "MZ") { fclose($h); return false; }
    // Offset do PE header em 0x3C (DWORD little-endian)
    $e_lfanew = unpack('V', substr($hdr, 0x3C, 4))[1] ?? null;
    if ($e_lfanew !== null && $e_lfanew > 0) {
        fseek($h, $e_lfanew);
        $pe = fread($h, 4);
        fclose($h);
        return ($pe === "PE\0\0");
    }
    fclose($h);
    return true; // Alguns empacotados podem esconder PE direto; "MZ" já indica executável DOS/PE.
}

/* -----------------------------
   ENDPOINT: META
------------------------------*/
$pdo = conDb();
ensure_schema($pdo);

if (isset($_GET['meta'])) {
    header('Content-Type: application/json; charset=utf-8');

    $active = pick_active_file();
    $meta = get_current_meta($active['path'], $active['rel']);
    $cfg  = load_cfg($pdo);

    $lastLog = null;
    if (!empty($cfg['last_log_id'])) {
        $stmt = $pdo->prepare("SELECT * FROM update_logs WHERE id=?");
        $stmt->execute([(int)$cfg['last_log_id']]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($row) $lastLog = $row;
    }

    // Proxy compacto host:port:user:pass
    $proxyCompact = '';
    if (!empty($cfg['proxy_host']) && !empty($cfg['proxy_port'])) {
        $proxyCompact = $cfg['proxy_host'].':'.$cfg['proxy_port'];
        if (!empty($cfg['proxy_user']) || !empty($cfg['proxy_pass'])) {
            $proxyCompact .= ':'.($cfg['proxy_user'] ?? '').':'.($cfg['proxy_pass'] ?? '');
        }
    }

    // Última atualização formatada BR
    $lastUpdatedBr = null;
    if (!empty($cfg['updated_at'])) {
        $ts = strtotime($cfg['updated_at']);
        if ($ts) $lastUpdatedBr = date('d/m/y H:i:s', $ts) . ' (horário de Brasília)';
    }

    echo json_encode([
        'file'           => $meta['file'],
        'present'        => $meta['present'],
        'size'           => $meta['size'],
        'sha256'         => $meta['sha256'],
        'last_modified'  => $meta['last_modified'],
        'last_updated'   => $lastUpdatedBr,
        'download_url'   => $cfg['download_url'] ?? '',
        'proxy_compact'  => $proxyCompact,
        'last_log'       => $lastLog ? [
                               'id'          => (int)$lastLog['id'],
                               'started_at'  => $lastLog['started_at'],
                               'finished_at' => $lastLog['finished_at'],
                               'ok'          => (int)$lastLog['ok'],
                               'messages'    => $lastLog['messages'],
                           ] : null
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/* -----------------------------
   ENDPOINT: UPDATE via LINK (POST, streaming texto)
------------------------------*/
if (isset($_GET['update'])) {
    header('Content-Type: text/plain; charset=utf-8');
    header('Cache-Control: no-store, no-cache, must-revalidate');
    header('X-Accel-Buffering: no');
    @ini_set('output_buffering', 'off');
    @ini_set('zlib.output_compression', '0');
    @ini_set('implicit_flush', '1');
    while (ob_get_level() > 0) { ob_end_flush(); }
    ob_implicit_flush(true);
    @set_time_limit(0);

    $url    = trim($_POST['download_url'] ?? '');
    $proxyIn= trim($_POST['proxy'] ?? '');

    $log = [];
    $t0 = time();
    $emit = function($line){ echo $line . "\n"; @ob_flush(); @flush(); };

    if ($url === '') {
        log_line($log, 'Erro: informe a URL do MSI.', $emit);
        $logId = persist_log($pdo, $log, false, $t0); save_cfg($pdo, ['last_log_id'=>$logId]); exit;
    }

    $proxy = parse_proxy($proxyIn);
    if ($proxyIn !== '' && !$proxy) {
        log_line($log, 'Erro: proxy em formato inválido. Use host:port:user:pass ou user:pass@host:port.', $emit);
        $logId = persist_log($pdo, $log, false, $t0); save_cfg($pdo, ['last_log_id'=>$logId]); exit;
    }

    // Salva configs iniciais (não mexe updated_at aqui)
    $setCfg = ['download_url'=>$url];
    if ($proxy) $setCfg += ['proxy_scheme'=>$proxy['scheme'],'proxy_host'=>$proxy['host'],'proxy_port'=>$proxy['port'],'proxy_user'=>$proxy['user'],'proxy_pass'=>$proxy['pass']];
    else $setCfg += ['proxy_scheme'=>null,'proxy_host'=>null,'proxy_port'=>null,'proxy_user'=>null,'proxy_pass'=>null];
    save_cfg($pdo, $setCfg);

    if ($proxy) { log_line($log, "Conectando ao proxy {$proxy['scheme']}://{$proxy['host']}:{$proxy['port']} …", $emit); detect_proxy_ip($proxy,$log,$emit); }
    else { log_line($log, 'Sem proxy. Conexão direta.', $emit); }

    log_line($log, "Baixando do link: {$url}", $emit);

    $tmp = $filePath.'.download.'.date('YmdHis').'.tmp';
    $ua  = get_dl_user_agent();
    $ok  = download_file($url, $tmp, $proxy, $log, $emit, $ua);
    if (!$ok) { $logId = persist_log($pdo, $log, false, $t0); save_cfg($pdo, ['last_log_id'=>$logId]); exit; }

    $size = @filesize($tmp) ?: 0;
    if ($size <= 0) { log_line($log, 'Arquivo baixado de tamanho 0.', $emit); @unlink($tmp); $logId = persist_log($pdo,$log,false,$t0); save_cfg($pdo,['last_log_id'=>$logId]); exit; }

    if (!is_probably_msi($tmp)) {
        log_line($log, 'Falha na validação: arquivo não parece MSI (assinatura OLE/CFB ausente).', $emit);
        @unlink($tmp); $logId = persist_log($pdo,$log,false,$t0); save_cfg($pdo,['last_log_id'=>$logId]); exit;
    }

    $hashNew = @hash_file('sha256', $tmp) ?: null;
    log_line($log, 'Hash (novo): '.($hashNew ?: '—'), $emit);

    $activeBefore = pick_active_file(); // qual estava ativo antes
    if ($activeBefore['present']) {
        log_line($log, 'Deletando arquivo anterior…', $emit);
        $bak = $activeBefore['path'].'.bak.'.date('YmdHis');
        @rename($activeBefore['path'], $bak);
        if (is_file($activeBefore['path'])) @unlink($activeBefore['path']);
        log_line($log, 'Anterior removido (backup: '.(is_file($bak)?basename($bak):'—').').', $emit);
    }

    // Posiciona como setup.msi
    $moved = @rename($tmp, MSI_PATH);
    if (!$moved) { log_line($log, 'Falha ao posicionar o novo MSI (rename).', $emit); @unlink($tmp); $logId = persist_log($pdo,$log,false,$t0); save_cfg($pdo,['last_log_id'=>$logId]); exit; }

    clearstatcache(true, MSI_PATH);
    $newMeta = get_current_meta(MSI_PATH, MSI_REL);

    $set = [
        'last_file'      => $activeBefore['rel'] ?? null,
        'last_size'      => $activeBefore['present'] ? (@filesize($activeBefore['path']) ?: null) : null,
        'last_sha256'    => $activeBefore['present'] ? (sha256_cached($activeBefore['path'])[0] ?? null) : null,
        'last_mtime'     => $activeBefore['present'] ? (@filemtime($activeBefore['path']) ?: null) : null,
        'current_file'   => $newMeta['file'],
        'current_size'   => $newMeta['size'],
        'current_sha256' => $newMeta['sha256'],
        'current_mtime'  => $newMeta['mtime'],
        'updated_at'     => date('Y-m-d H:i:s'),
    ];
    save_cfg($pdo, $set);

    log_line($log, 'Novo MSI atualizado com sucesso em '.date('d/m/y \à\s H:i:s').' (horário de Brasília).', $emit);

    $logId = persist_log($pdo, $log, true, $t0);
    save_cfg($pdo, ['last_log_id'=>$logId]);

    echo "--- FIM ---\n";
    @ob_flush(); @flush();
    exit;
}

/* -----------------------------
   ENDPOINT: UPLOAD (POST, JSON)
   - Campo do arquivo: "package"
------------------------------*/
if (isset($_GET['upload'])) {
    header('Content-Type: application/json; charset=utf-8');

    $log = [];
    $t0 = time();

    if (!isset($_FILES['package']) || empty($_FILES['package']['tmp_name'])) {
        $log[] = '['.date('d/m/y H:i:s').'] Erro: nenhum arquivo recebido.';
        echo json_encode(['ok'=>false,'messages'=>implode("\n",$log)]); exit;
    }

    $err = $_FILES['package']['error'];
    if ($err !== UPLOAD_ERR_OK) {
        $log[] = '['.date('d/m/y H:i:s').'] Erro no upload (código '.$err.'). Verifique upload_max_filesize/post_max_size.';
        echo json_encode(['ok'=>false,'messages'=>implode("\n",$log)]); exit;
    }

    $tmpIn = $_FILES['package']['tmp_name'];
    $size  = (int)($_FILES['package']['size'] ?? 0);
    $orig  = $_FILES['package']['name'] ?? 'arquivo';

    $log[] = '['.date('d/m/y H:i:s').'] Upload recebido: '.$orig.' ('.number_format($size/1024,1,',','.').' KB).';

    // Copia para tmp próprio para evitar problemas de permissão
    $tmp = __DIR__ . '/upload.' . date('YmdHis') . '.tmp';
    if (!@move_uploaded_file($tmpIn, $tmp)) {
        // fallback: copy
        if (!@copy($tmpIn, $tmp)) {
            $log[] = '['.date('d/m/y H:i:s').'] Falha ao preparar arquivo de upload.';
            echo json_encode(['ok'=>false,'messages'=>implode("\n",$log)]); exit;
        }
    }

    // Detecta tipo
    $isMsi = is_probably_msi($tmp);
    $isExe = !$isMsi && is_probably_exe($tmp);

    if (!$isMsi && !$isExe) {
        @unlink($tmp);
        $log[] = '['.date('d/m/y H:i:s').'] Falha na validação: arquivo não parece MSI/EXE válido.';
        echo json_encode(['ok'=>false,'messages'=>implode("\n",$log)]); exit;
    }

    $destRel  = $isMsi ? MSI_REL : EXE_REL;
    $destPath = $isMsi ? MSI_PATH : EXE_PATH;
    $log[] = '['.date('d/m/y H:i:s').'] Detectado tipo: '.($isMsi?'MSI':'EXE').'. Será salvo como '.$destRel.'.';

    // Quem estava ativo antes (msi ou exe)
    $activeBefore = pick_active_file();

    // Backup do arquivo com MESMO nome de destino, se existir
    if (is_file($destPath)) {
        $bak = $destPath.'.bak.'.date('YmdHis');
        @rename($destPath, $bak);
        if (is_file($destPath)) @unlink($destPath);
        $log[] = '['.date('d/m/y H:i:s').'] Arquivo anterior ('.$destRel.') movido para '.basename($bak).'.';
    }

    // Move para destino
    if (!@rename($tmp, $destPath)) {
        @unlink($tmp);
        $log[] = '['.date('d/m/y H:i:s').'] Falha ao posicionar o novo arquivo (rename).';
        echo json_encode(['ok'=>false,'messages'=>implode("\n",$log)]); exit;
    }

    clearstatcache(true, $destPath);
    $newMeta = get_current_meta($destPath, $destRel);
    $hashNew = $newMeta['sha256'] ?: (@hash_file('sha256', $destPath) ?: null);
    if ($hashNew && !$newMeta['sha256']) { // atualiza cache
        @file_put_contents($destPath.'.sha256.json', json_encode(['mtime'=>$newMeta['mtime'],'sha256'=>$hashNew], JSON_UNESCAPED_SLASHES));
    }
    $log[] = '['.date('d/m/y H:i:s').'] Hash (novo): '.($hashNew ?: '—');

    // Atualiza config
    $set = [
        'last_file'      => $activeBefore['rel'] ?? null,
        'last_size'      => $activeBefore['present'] ? (@filesize($activeBefore['path']) ?: null) : null,
        'last_sha256'    => $activeBefore['present'] ? (sha256_cached($activeBefore['path'])[0] ?? null) : null,
        'last_mtime'     => $activeBefore['present'] ? (@filemtime($activeBefore['path']) ?: null) : null,
        'current_file'   => $newMeta['file'],
        'current_size'   => $newMeta['size'],
        'current_sha256' => $hashNew,
        'current_mtime'  => $newMeta['mtime'],
        'updated_at'     => date('Y-m-d H:i:s'),
    ];
    save_cfg($pdo, $set);

    $log[] = '['.date('d/m/y H:i:s').'] Novo '.($isMsi?'MSI':'EXE').' atualizado com sucesso em '.date('d/m/y \à\s H:i:s').' (horário de Brasília).';

    // Persiste log
    $logId = persist_log($pdo, $log, true, $t0);
    save_cfg($pdo, ['last_log_id'=>$logId]);

    echo json_encode(['ok'=>true, 'messages'=>implode("\n",$log)], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/* -----------------------------
   HTML
------------------------------*/
header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Atualizador de MSI</title>
  <meta name="color-scheme" content="light dark">
  <style>
    :root{
      --bg-1:#0b1b3a; --bg-2:#0f2247; --card:#0b1220e6; --muted:#9fb0c8; --text:#e8eef7; --title:#f8fbff;
      --brand:#0b3b8c; --brand-2:#1d4ed8; --brand-3:#3b82f6; --ok:#18a86b; --warn:#eab308; --err:#ef4444;
      --ring:#ffffff24; --shadow:0 20px 45px #00000055; --radius:16px;
    }
    @media (prefers-color-scheme: light){
      :root{ --bg-1:#f0f4fb; --bg-2:#e6eef9; --card:#fffffff2; --muted:#5b6b85; --text:#0f172a; --title:#0b1220; --ring:#00000012; --shadow:0 20px 45px #0b122015; }
    }
    *{box-sizing:border-box}
    html,body{height:100%;margin:0;color:var(--text);font-family:ui-sans-serif,system-ui,-apple-system,Segoe UI,Roboto,"Helvetica Neue",Arial}
    body{
      background:
        radial-gradient(1000px 700px at 12% 18%, color-mix(in oklab, var(--brand-2) 22%, transparent) 0, transparent 60%),
        radial-gradient(900px 640px at 88% 28%, color-mix(in oklab, var(--brand) 24%, transparent) 0, transparent 60%),
        linear-gradient(180deg, var(--bg-1), var(--bg-2));
      position:relative; display:grid; place-items:center; overflow:hidden;
    }
    body::before{content:"";position:absolute;inset:-20%;background:conic-gradient(from 180deg at 50% 50%,#fff8 0,transparent 60deg,#fff6 120deg,transparent 180deg,#fff8 240deg,transparent 300deg,#fff6 360deg);filter:blur(40px) saturate(120%);pointer-events:none;opacity:.5;}
    body::after{content:"";position:absolute;inset:0;pointer-events:none;opacity:.06;background-image:repeating-linear-gradient(45deg,#000 0 1px,transparent 1px 4px),repeating-linear-gradient(135deg,#000 0 1px,transparent 1px 4px);mix-blend-mode:soft-light}

    .wrap{width:min(880px,94vw);background:var(--card);backdrop-filter:saturate(130%) blur(10px);
      border:1px solid var(--ring);border-radius:var(--radius);box-shadow:var(--shadow);overflow:hidden;position:relative}

    .header{display:grid;place-items:center;gap:10px;padding:24px 26px;
      background:linear-gradient(135deg,color-mix(in oklab,var(--brand) 60%,transparent),color-mix(in oklab,var(--brand-2) 45%,transparent));
      border-bottom:1px solid var(--ring)}
    .h1{font-weight:800;letter-spacing:.2px;color:var(--title);font-size:clamp(1.1rem,.8rem+1.4vw,1.5rem);text-align:center}
    .sub{color:var(--muted);font-size:.95rem;text-align:center}

    .body{padding:26px clamp(18px,4vw,26px);display:grid;gap:18px}
    .status{display:flex;align-items:center;justify-content:space-between;gap:12px;background:linear-gradient(180deg,#ffffff08,transparent);
      border:1px dashed var(--ring);border-radius:12px;padding:12px 14px;font-size:.95rem}
    .status .left{display:flex;align-items:center;gap:10px}
    .dot{width:10px;height:10px;border-radius:99px;background: var(--warn);box-shadow:0 0 0 6px #f59e0b22}

    .meta{display:grid;gap:12px;font-size:.92rem;grid-template-columns:1fr}
    @media (min-width:760px){ .meta{grid-template-columns:1fr 1fr} }
    .meta div{min-width:0;padding:10px 12px;border:1px solid var(--ring);border-radius:10px;background:#ffffff07}
    .meta b{display:block;font-size:.78rem;color:var(--muted);font-weight:600;letter-spacing:.2px;text-transform:uppercase}
    .meta span{display:block;margin-top:6px;overflow-wrap:anywhere;word-break:break-word}
    .meta > :nth-child(4) span{font-family:ui-monospace,SFMono-Regular,Menlo,Monaco,Consolas,"Liberation Mono",monospace;font-size:.86rem;line-height:1.2}

    .tabs{display:flex;border:1px solid var(--ring);border-radius:10px;overflow:hidden}
    .tab{flex:1;padding:10px 12px;text-align:center;cursor:pointer;background:#ffffff0a;color:var(--title)}
    .tab.active{background:linear-gradient(135deg,color-mix(in oklab,var(--brand-2),white 28%),color-mix(in oklab,var(--brand-3),white 18%))}
    .pane{display:none;border:1px solid var(--ring);border-radius:10px;padding:14px;background:#ffffff07}
    .pane.active{display:block}

    .cfg{display:grid;gap:10px}
    .field{display:grid;gap:6px}
    .field label{font-size:.85rem;color:var(--muted)}
    .field input, .field input[type=file]{width:100%;padding:10px 12px;border-radius:10px;background:#ffffff10;border:1px solid var(--ring);color:var(--title)}
    .field input::placeholder{color:#9fb0c8aa}

    .console{width:100%;height:220px;padding:12px;border-radius:10px;background:#0c0f14;border:1px solid #232a38;color:#bfe4ff;
      font-family:ui-monospace,SFMono-Regular,Menlo,Consolas,monospace;font-size:.9rem;line-height:1.35;white-space:pre;overflow:auto}

    .actions{display:flex;flex-wrap:wrap;gap:10px}
    .btn{appearance:none;border:none;cursor:pointer;padding:10px 14px;border-radius:12px;font-weight:600;letter-spacing:.2px;
      display:inline-flex;align-items:center;gap:10px;background:#ffffff15;color:var(--title);border:1px solid var(--ring);
      box-shadow:0 6px 16px #00000030;transition:transform .12s ease, box-shadow .12s ease, background .2s ease;text-decoration:none}
    .btn:hover{transform:translateY(-1px);box-shadow:0 10px 22px #00000035}
    .btn.primary{background:linear-gradient(135deg,color-mix(in oklab,var(--brand-2),white 28%),color-mix(in oklab,var(--brand-3),white 18%));border-color:#ffffff66}
    .btn.primary:hover{filter:brightness(1.08)}
    .btn.ghost{background:transparent}
    .btn[disabled]{opacity:.65;cursor:not-allowed}

    .footer{padding:16px 26px;border-top:1px solid var(--ring);display:flex;align-items:center;justify-content:space-between;gap:10px;color:var(--muted);font-size:.9rem}
  </style>
</head>
<body>
  <main class="wrap" role="main" aria-labelledby="title">
    <header class="header">
      <div id="title" class="h1">Atualizador de MSI</div>
      <div class="sub">Baixe/substitua via link ou faça upload do instalador — com proxy e validação.</div>
    </header>

    <section class="body">
      <div class="status">
        <div class="left">
          <span class="dot" id="dot"></span>
          <span id="statusText">Carregando dados…</span>
        </div>
        <span id="pct">—</span>
      </div>

      <div class="status">
        <div class="left">Última atualização: <span id="lastUpdated"></span></div>
        <span></span>
      </div>

      <div class="meta" aria-label="Detalhes do arquivo">
        <div><b>Arquivo</b>       <span id="fileName">—</span></div>
        <div><b>Tamanho</b>       <span id="fileSize">—</span></div>
        <div><b>Modificado em</b> <span id="fileDate">—</span></div>
        <div><b>Hash</b>          <span id="hash">—</span></div>
      </div>

      <!-- TABS -->
      <div class="tabs" role="tablist" aria-label="Forma de atualização">
        <button class="tab active" id="tab-link"   type="button" role="tab" aria-selected="true">Atualizar via Link</button>
        <button class="tab"         id="tab-upload" type="button" role="tab" aria-selected="false">Atualizar via Upload</button>
      </div>

      <!-- PANE: LINK -->
      <div class="pane active" id="pane-link" role="tabpanel" aria-labelledby="tab-link">
        <div class="cfg">
          <div class="field">
            <label for="download_url">Link de download do MSI</label>
            <input id="download_url" type="url" placeholder="https://exemplo.com/installer.msi" />
          </div>
          <div class="field">
            <label for="proxy">Proxy (opcional — host:port:user:pass | user:pass@host:port | http://user:pass@host:port | socks5://host:port:user:pass)</label>
            <input id="proxy" type="text" placeholder="geo.iproyal.com:11210:login:senha" />
          </div>
        </div>
        <div class="actions" style="margin-top:10px">
          <button class="btn primary" id="btnUpdate">
            <svg width="18" height="18" viewBox="0 0 24 24" aria-hidden="true"><path fill="currentColor" d="M12 3v5H7l5 7 5-7h-5V3zM5 19h14v2H5z"/></svg>
            Baixar e atualizar pelo Link
          </button>
          <a class="btn ghost" href="#" id="btnReload">Recarregar dados</a>
        </div>
      </div>

      <!-- PANE: UPLOAD -->
      <div class="pane" id="pane-upload" role="tabpanel" aria-labelledby="tab-upload">
        <div class="cfg">
          <div class="field">
            <label for="upload_file">Selecione o arquivo (.msi ou .exe)</label>
            <input id="upload_file" type="file" accept=".msi,.exe,application/x-msdownload,application/x-ms-installer,application/octet-stream" />
          </div>
        </div>
        <div class="actions" style="margin-top:10px">
          <button class="btn primary" id="btnUpload">
            <svg width="18" height="18" viewBox="0 0 24 24" aria-hidden="true"><path fill="currentColor" d="M5 20h14v-2H5v2zm7-18L5.33 8h3.84v6h4.66V8h3.84L12 2z"/></svg>
            Enviar e atualizar por Upload
          </button>
        </div>
      </div>

      <div>
        <b class="sub">Console (log da última/atual atualização)</b>
        <textarea class="console" id="console" readonly>—</textarea>
      </div>
    </section>

    <footer class="footer" aria-label="Rodapé">
      <div>By GPTr</div>
      <div>&nbsp;</div>
    </footer>
  </main>

  <script>
    const metaUrl    = '?meta=1';
    const updateUrl  = '?update=1';
    const uploadUrl  = '?upload=1';

    const fileNameEl = document.getElementById('fileName');
    const fileSizeEl = document.getElementById('fileSize');
    const fileDateEl = document.getElementById('fileDate');
    const hashEl     = document.getElementById('hash');
    const lastUpdEl  = document.getElementById('lastUpdated');
    const statusText = document.getElementById('statusText');
    const pct        = document.getElementById('pct');
    const dot        = document.getElementById('dot');
    const consoleEl  = document.getElementById('console');

    const downloadUrlInput = document.getElementById('download_url');
    const proxyInput       = document.getElementById('proxy');
    const btnUpdate        = document.getElementById('btnUpdate');

    const tabLink   = document.getElementById('tab-link');
    const tabUpload = document.getElementById('tab-upload');
    const paneLink  = document.getElementById('pane-link');
    const paneUpload= document.getElementById('pane-upload');
    const btnReload = document.getElementById('btnReload');
    const fileInput = document.getElementById('upload_file');
    const btnUpload = document.getElementById('btnUpload');

    function nowStr(){
      const d = new Date();
      const dd = String(d.getDate()).padStart(2,'0');
      const mm = String(d.getMonth()+1).padStart(2,'0');
      const yy = String(d.getFullYear()).slice(-2);
      const hh = String(d.getHours()).padStart(2,'0');
      const mi = String(d.getMinutes()).padStart(2,'0');
      const ss = String(d.getSeconds()).padStart(2,'0');
      return `[${dd}/${mm}/${yy} ${hh}:${mi}:${ss}]`;
    }

    function formatBytes(bytes){
      if (!bytes && bytes !== 0) return '—';
      const u = ['B','KB','MB','GB','TB']; let i = 0; let n = bytes;
      while(n >= 1024 && i < u.length-1){ n/=1024; i++; }
      return n.toFixed(n < 10 ? 2 : 1) + ' ' + u[i];
    }

    function appendConsole(text){
      if (consoleEl.value === '—') consoleEl.value = '';
      consoleEl.value += text;
      consoleEl.scrollTop = consoleEl.scrollHeight;
      const m = text.match(/Download (\d+)% concluído/);
      if (m){
        pct.textContent = m[1] + '%';
        dot.style.background = 'var(--brand-3)';
        statusText.textContent = 'Baixando…';
      }
      if (/atualizado com sucesso/.test(text)){
        dot.style.background = 'var(--ok)';
        statusText.textContent = 'Atualizado com sucesso.';
        pct.textContent = '100%';
      }
      if (/Erro|Falha/.test(text)){
        dot.style.background = 'var(--err)';
        statusText.textContent = 'Falha na atualização.';
      }
    }

    async function loadMeta(){
      statusText.textContent = 'Carregando dados…';
      pct.textContent = '—';
      dot.style.background = 'var(--warn)';
      try{
        const res = await fetch(metaUrl, {cache:'no-store'});
        if(!res.ok) throw new Error('HTTP '+res.status);
        const m = await res.json();
        fileNameEl.textContent = m.file || '—';
        fileSizeEl.textContent = m.size ? formatBytes(m.size) : '—';
        fileDateEl.textContent = m.last_modified || '—';
        hashEl.textContent     = m.sha256 ? ('sha256:' + m.sha256) : '—';
        lastUpdEl.textContent  = m.last_updated || '—';

        if (m.download_url) downloadUrlInput.value = m.download_url;
        if (m.proxy_compact) proxyInput.value = m.proxy_compact;

        if (m.last_log && m.last_log.messages){
          consoleEl.value = m.last_log.messages + (m.last_log.messages.endsWith('\n')?'':'\n');
          consoleEl.scrollTop = consoleEl.scrollHeight;
        } else {
          consoleEl.value = '—';
        }
        dot.style.background = m.present ? 'var(--ok)' : 'var(--warn)';
        statusText.textContent = m.present ? 'Arquivo atual encontrado.' : 'Nenhum instalador encontrado.';
        pct.textContent = '';
      }catch(e){
        statusText.textContent = 'Erro ao ler metadados.';
        console.error(e);
      }
    }

    async function runUpdate(){
      const url = downloadUrlInput.value.trim();
      const proxy = proxyInput.value.trim();
      if (!url){ alert('Informe o link do MSI.'); return; }

      btnUpdate.disabled = true;
      const oldBtnHtml = btnUpdate.innerHTML;
      btnUpdate.innerHTML = 'Baixando pelo Link …';

      statusText.textContent = 'Iniciando atualização…';
      pct.textContent = '…';
      dot.style.background = 'var(--brand-3)';

      const fd = new FormData();
      fd.append('download_url', url);
      fd.append('proxy', proxy);

      try{
        const res = await fetch(updateUrl, { method:'POST', body: fd });
        if (!res.ok || !res.body){
          appendConsole(`${nowStr()} Erro de resposta do servidor (HTTP ${res.status}).\n`);
          return;
        }
        consoleEl.value = '';
        const reader = res.body.getReader();
        const decoder = new TextDecoder();
        while (true){
          const {value, done} = await reader.read();
          if (done) break;
          const chunk = decoder.decode(value, {stream:true});
          appendConsole(chunk);
        }
        await loadMeta();
      }catch(e){
        appendConsole(`${nowStr()} Erro na requisição: ${String(e)}\n`);
        dot.style.background = 'var(--err)';
        statusText.textContent = 'Erro na requisição.';
        pct.textContent = 'X';
      }finally{
        btnUpdate.disabled = false;
        btnUpdate.innerHTML = oldBtnHtml;
      }
    }

    function runUpload(){
      const f = fileInput.files && fileInput.files[0];
      if (!f){ alert('Escolha um arquivo .msi ou .exe.'); return; }

      btnUpload.disabled = true;
      const oldTxt = btnUpload.innerHTML;
      btnUpload.innerHTML = 'Enviando arquivo …';

      // Zera barra/estado
      statusText.textContent = 'Enviando arquivo…';
      pct.textContent = '0%';
      dot.style.background = 'var(--brand-3)';
      if (consoleEl.value === '—') consoleEl.value = '';

      const fd = new FormData();
      fd.append('package', f);

      const xhr = new XMLHttpRequest();
      xhr.open('POST', uploadUrl, true);

      let lastPct = -10;
      xhr.upload.onprogress = (e) => {
        if (e.lengthComputable){
          const p = Math.floor((e.loaded / e.total) * 100);
          if (p >= lastPct + 5 || p === 100){
            lastPct = p;
            appendConsole(`${nowStr()} Upload ${p}% concluído…\n`);
            pct.textContent = p + '%';
          }
        } else {
          appendConsole(`${nowStr()} Enviando…\n`);
        }
      };

      xhr.onerror = () => {
        appendConsole(`${nowStr()} Erro de rede durante upload.\n`);
        dot.style.background = 'var(--err)';
        statusText.textContent = 'Falha no upload.';
        btnUpload.disabled = false;
        btnUpload.innerHTML = oldTxt;
      };

      xhr.onload = async () => {
        btnUpload.disabled = false;
        btnUpload.innerHTML = oldTxt;

        try{
          const data = JSON.parse(xhr.responseText || '{}');
          if (!data.ok){
            appendConsole((data.messages ? data.messages : `${nowStr()} Erro no processamento do upload.`) + '\n');
            dot.style.background = 'var(--err)';
            statusText.textContent = 'Falha na atualização.';
          } else {
            appendConsole((data.messages || '') + (String(data.messages).endsWith('\n')?'':'\n'));
            dot.style.background = 'var(--ok)';
            statusText.textContent = 'Atualizado com sucesso.';
            pct.textContent = '100%';
            await loadMeta();
          }
        }catch(e){
          appendConsole(`${nowStr()} Resposta inválida do servidor.\n`);
          dot.style.background = 'var(--err)';
          statusText.textContent = 'Falha na atualização.';
        }
      };

      xhr.send(fd);
    }

    // Tabs
    function activateTab(which){
      if (which === 'link'){
        tabLink.classList.add('active'); tabUpload.classList.remove('active');
        paneLink.classList.add('active'); paneUpload.classList.remove('active');
        tabLink.setAttribute('aria-selected','true'); tabUpload.setAttribute('aria-selected','false');
      } else {
        tabUpload.classList.add('active'); tabLink.classList.remove('active');
        paneUpload.classList.add('active'); paneLink.classList.remove('active');
        tabUpload.setAttribute('aria-selected','true'); tabLink.setAttribute('aria-selected','false');
      }
    }
    tabLink.addEventListener('click', ()=>activateTab('link'));
    tabUpload.addEventListener('click', ()=>activateTab('upload'));

    // Actions
    btnUpdate.addEventListener('click', runUpdate);
    btnReload.addEventListener('click', (e)=>{ e.preventDefault(); loadMeta(); });
    btnUpload.addEventListener('click', runUpload);

    // Boot
    window.addEventListener('load', loadMeta);
  </script>
</body>
</html>
