import sys
import ipaddress
from itertools import islice

def is_subnet_of(subnet, network):
    subnet = ipaddress.ip_network(subnet)
    network = ipaddress.ip_network(network)
    return subnet != network and network.supernet_of(subnet)

if len(sys.argv) != 2:
    print("Usage: python3.7 clean_networks.py <input_file>")
    sys.exit(1)

input_file = sys.argv[1]

try:
    with open(input_file, 'r') as f:
        networks = f.readlines()
except FileNotFoundError:
    print(f"File not found: {input_file}")
    sys.exit(1)

print(f"Read {len(networks)} lines from {input_file}")

unique_networks = set()
for i, network in enumerate(networks):
    network = network.strip()
    if network.startswith('#') or not network:
        continue  # Skip comments and empty lines
    parts = network.split()
    if len(parts) < 2:
        continue  # Skip lines that don't match the expected format
    cidr = parts[0].strip()
    try:
        ipaddress.ip_network(cidr)
    except ValueError:
        continue  # Skip invalid network addresses
    unique_networks.add(cidr)
    if i % 100 == 0:
        print(f"Processed {i} lines, current unique networks count: {len(unique_networks)}")

print(f"Total unique networks: {len(unique_networks)}")

cleaned_networks = set(unique_networks)
unique_networks_list = list(unique_networks)
for i, network in enumerate(unique_networks_list):
    is_subnet = False
    for j, other in enumerate(unique_networks_list):
        if network != other and is_subnet_of(network, other):
            is_subnet = True
            break
    if is_subnet:
        cleaned_networks.discard(network)
    if i % 100 == 0:
        print(f"Checked {i} networks, cleaned networks count: {len(cleaned_networks)}")

output_file = 'cleaned_' + input_file

try:
    with open(output_file, 'w') as f:
        for network in cleaned_networks:
            f.write(f"{network} 1;\n")
    print(f"Cleaned network list has been saved to {output_file}")
except Exception as e:
    print(f"Error writing to file: {e}")
